<?php
// config.php - Database and JWT/Basic Token settings (manual inclusion with fallback)
require_once 'C:/xampp/htdocs/chat/jwt/JWT.php';
require_once 'C:/xampp/htdocs/chat/jwt/Key.php';
require_once 'C:/xampp/htdocs/chat/jwt/JWTExceptionWithPayloadInterface.php';
require_once 'C:/xampp/htdocs/chat/jwt/BeforeValidException.php';
require_once 'C:/xampp/htdocs/chat/jwt/ExpiredException.php';
require_once 'C:/xampp/htdocs/chat/jwt/SignatureInvalidException.php';

use Firebase\JWT\JWT;
use Firebase\JWT\Key;

// Configuration
define('DB_HOST', 'localhost');
define('DB_USER', 'root');
define('DB_PASS', '');
define('DB_NAME', 'chat');
define('JWT_SECRET', 'your_jwt_secret_key_very_long_and_random_2025');
define('TOKEN_SECRET', 'your_token_secret_key_very_long_and_random_2025'); // For basic token
define('USE_JWT', false); // Set to true for JWT, false for basic token
define('UPLOAD_DIR', 'C:/xampp/htdocs/chat/uploads/');
define('USER_DATA_DIR', UPLOAD_DIR . 'user_data/');
define('CHAT_UPLOAD_DIR', UPLOAD_DIR . 'chats/');
define('RATE_LIMIT_REQUESTS', 100);
define('RATE_LIMIT_WINDOW', 3600);

// Create directories
if (!file_exists(UPLOAD_DIR)) mkdir(UPLOAD_DIR, 0755, true);
if (!file_exists(USER_DATA_DIR)) mkdir(USER_DATA_DIR, 0755, true);
if (!file_exists(CHAT_UPLOAD_DIR)) mkdir(CHAT_UPLOAD_DIR, 0755, true);

// Database connection
function getDB() {
    try {
        $pdo = new PDO("mysql:host=" . DB_HOST . ";dbname=" . DB_NAME, DB_USER, DB_PASS);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        return $pdo;
    } catch (PDOException $e) {
        http_response_code(500);
        die(json_encode(['error' => 'DB Connection failed: ' . $e->getMessage()]));
    }
}

// JWT functions (if enabled)
function generateJWT($user_id) {
    if (!USE_JWT) return generateToken($user_id); // Fallback to basic token
    $payload = ['iss' => 'localhost', 'iat' => time(), 'exp' => time() + 86400, 'sub' => $user_id];
    return JWT::encode($payload, JWT_SECRET, 'HS256');
}

function verifyJWT($token) {
    if (!USE_JWT) return verifyToken(getDB(), $token); // Fallback to basic token
    try {
        $decoded = JWT::decode($token, new Key(JWT_SECRET, 'HS256'));
        return $decoded->sub;
    } catch (Exception $e) {
        return false;
    }
}

// Basic token functions (fallback)
function generateToken($user_id) {
    return hash('sha256', $user_id . time() . TOKEN_SECRET);
}

function verifyToken($pdo, $token) {
    $stmt = $pdo->prepare("SELECT id FROM users WHERE token = :token");
    $stmt->execute(['token' => $token]);
    return $stmt->fetch() ? $stmt->fetch()['id'] : false;
}

// Rate limiting
function checkRateLimit($pdo, $user_id, $endpoint) {
    $stmt = $pdo->prepare("SELECT request_count, last_request FROM rate_limits WHERE user_id = :user_id AND endpoint = :endpoint");
    $stmt->execute(['user_id' => $user_id, 'endpoint' => $endpoint]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);

    $now = time();
    if ($row) {
        $last_request = strtotime($row['last_request']);
        if ($now - $last_request < RATE_LIMIT_WINDOW) {
            if ($row['request_count'] >= RATE_LIMIT_REQUESTS) {
                http_response_code(429);
                die(json_encode(['error' => 'Rate limit exceeded']));
            }
            $stmt = $pdo->prepare("UPDATE rate_limits SET request_count = request_count + 1, last_request = NOW() WHERE user_id = :user_id AND endpoint = :endpoint");
            $stmt->execute(['user_id' => $user_id, 'endpoint' => $endpoint]);
        } else {
            $stmt = $pdo->prepare("UPDATE rate_limits SET request_count = 1, last_request = NOW() WHERE user_id = :user_id AND endpoint = :endpoint");
            $stmt->execute(['user_id' => $user_id, 'endpoint' => $endpoint]);
        }
    } else {
        $stmt = $pdo->prepare("INSERT INTO rate_limits (user_id, endpoint, request_count, last_request) VALUES (:user_id, :endpoint, 1, NOW())");
        $stmt->execute(['user_id' => $user_id, 'endpoint' => $endpoint]);
    }
}