<?php
require_once '../config/database.php';

handleCors();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendResponse(['error' => 'Method not allowed'], 405);
}

// Check if it's a file upload or message
$is_file_upload = isset($_FILES) && !empty($_FILES);

try {
    $database = new Database();
    $pdo = $database->getConnection();
    
    if (!$pdo) {
        throw new Exception('Database connection failed');
    }
    
    // Get user info from token or email
    $user_email = $_POST['user_email'] ?? null;
    $token = $_POST['token'] ?? null;
    
    if (!$user_email) {
        sendResponse(['error' => 'User email is required'], 400);
    }
    
    // Get user ID
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$user_email]);
    $user = $stmt->fetch();
    
    if (!$user) {
        sendResponse(['error' => 'User not found'], 404);
    }
    
    $user_id = $user['id'];
    $user_folder = createUserFolder($user_email);
    
    if ($is_file_upload) {
        // Handle file upload (images, voice messages, videos, documents)
        $uploaded_files = [];
        $total_files = count($_FILES);
        
        foreach ($_FILES as $field_name => $file) {
            if ($file['error'] !== UPLOAD_ERR_OK) {
                continue;
            }
            
            $file_name = $file['name'];
            $file_tmp = $file['tmp_name'];
            $file_size = $file['size'];
            $file_type = $file['type'];
            
            // Determine file category and type
            $file_extension = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
            $category = 'chat';
            
            if (in_array($file_extension, ['jpg', 'jpeg', 'png', 'gif', 'webp'])) {
                $db_file_type = 'image';
                $subfolder = 'images';
            } elseif (in_array($file_extension, ['mp4', 'avi', 'mov', 'mkv', 'webm'])) {
                $db_file_type = 'video';
                $subfolder = 'videos';
            } elseif (in_array($file_extension, ['mp3', 'wav', 'aac', 'm4a', 'ogg'])) {
                $db_file_type = 'audio';
                $subfolder = 'voice_messages';
            } elseif (in_array($file_extension, ['pdf', 'doc', 'docx', 'txt', 'xls', 'xlsx'])) {
                $db_file_type = 'document';
                $subfolder = 'documents';
            } else {
                $db_file_type = 'other';
                $subfolder = 'other';
            }
            
            // Check if it's a voice message based on field name or duration
            if (strpos($field_name, 'voice') !== false || isset($_POST['is_voice_message'])) {
                $db_file_type = 'voice_message';
                $subfolder = 'voice_messages';
            }
            
            // Create unique filename to prevent conflicts
            $unique_filename = time() . '_' . uniqid() . '_' . $file_name;
            $upload_path = $user_folder . '/chat/' . $subfolder;
            
            if (!file_exists($upload_path)) {
                mkdir($upload_path, 0755, true);
            }
            
            $file_path = $upload_path . '/' . $unique_filename;
            
            if (move_uploaded_file($file_tmp, $file_path)) {
                // Calculate file hash for duplicate detection
                $file_hash = calculateFileHash($file_path);
                
                // Check if file already exists for this user
                $stmt = $pdo->prepare("SELECT id FROM files WHERE user_id = ? AND file_hash = ?");
                $stmt->execute([$user_id, $file_hash]);
                $existing_file = $stmt->fetch();
                
                if ($existing_file) {
                    // File already exists, remove the duplicate upload
                    unlink($file_path);
                    $uploaded_files[] = [
                        'success' => true,
                        'message' => 'File already exists (duplicate)',
                        'file_id' => $existing_file['id'],
                        'duplicate' => true
                    ];
                } else {
                    // Insert file record
                    $stmt = $pdo->prepare("
                        INSERT INTO files (user_id, file_name, file_path, file_type, file_size, file_hash, mime_type, category, upload_status, upload_progress, created_at) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'completed', 100, NOW())
                    ");
                    $stmt->execute([$user_id, $file_name, $file_path, $db_file_type, $file_size, $file_hash, $file_type, $category]);
                    
                    $file_id = $pdo->lastInsertId();
                    
                    $uploaded_files[] = [
                        'success' => true,
                        'file_id' => $file_id,
                        'file_name' => $file_name,
                        'file_path' => $file_path,
                        'file_type' => $db_file_type,
                        'file_size' => $file_size,
                        'duplicate' => false
                    ];
                }
            } else {
                $uploaded_files[] = [
                    'success' => false,
                    'error' => 'Failed to upload file: ' . $file_name
                ];
            }
        }
        
        sendResponse([
            'success' => true,
            'message' => 'Files processed',
            'files' => $uploaded_files,
            'total_files' => $total_files
        ]);
        
    } else {
        // Handle text message or file reference
        $message_data = getRequestData();
        
        if (!$message_data) {
            sendResponse(['error' => 'Invalid request data'], 400);
        }
        
        $message_type = $message_data['message_type'] ?? 'text';
        $message_content = $message_data['message_content'] ?? '';
        $chat_room = $message_data['chat_room'] ?? 'general';
        $sender_email = $user_email;
        $receiver_email = $message_data['receiver_email'] ?? null;
        $file_id = $message_data['file_id'] ?? null;
        
        // Insert chat message
        $stmt = $pdo->prepare("
            INSERT INTO chat_messages (user_id, message_type, message_content, file_id, chat_room, sender_email, receiver_email, is_sent, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, TRUE, NOW())
        ");
        $stmt->execute([$user_id, $message_type, $message_content, $file_id, $chat_room, $sender_email, $receiver_email]);
        
        $message_id = $pdo->lastInsertId();
        
        sendResponse([
            'success' => true,
            'message' => 'Message sent successfully',
            'message_id' => $message_id,
            'message_type' => $message_type,
            'chat_room' => $chat_room
        ]);
    }
    
} catch(Exception $e) {
    logError('Chat Files API Error: ' . $e->getMessage());
    sendResponse(['error' => 'Internal server error: ' . $e->getMessage()], 500);
}
?>