<?php
require_once '../config/database.php';


handleCors();

$method = $_SERVER['REQUEST_METHOD'];

switch ($method) {
    case 'GET':
        getUserActions();
        break;
    case 'POST':
        updateUserActions();
        break;
    case 'PUT':
        updateLocationTracking();
        break;
    default:
        sendResponse(['error' => 'Method not allowed'], 405);
}

function getUserActions() {
    try {
        $headers = getallheaders();
        if ($headers === false) {
             $headers = [];
             foreach ($_SERVER as $name => $value) {
                if (substr($name, 0, 5) == 'HTTP_') {
                    $headerKey = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($name, 5)))));
                    $headers[$headerKey] = $value;
                }
            }
        }
        
        $user_id = getUserIdFromHeaders($headers);
        
        if (!$user_id) {
            sendResponse(['error' => 'User not authenticated'], 401);
            return;
        }
        
        $database = new Database();
        $pdo = $database->getConnection();
        
        $stmt = $pdo->prepare("SELECT * FROM actions WHERE user_id = ?");
        $stmt->execute([$user_id]);
        $actions = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$actions) {
            // Create default actions record for user
            $stmt = $pdo->prepare("INSERT INTO actions (user_id, isCamRecording, isVoiceRecording, isLocationLive, isOnline) VALUES (?, 0, 0, 0, 0)");
            $stmt->execute([$user_id]);
            
            $stmt = $pdo->prepare("SELECT * FROM actions WHERE user_id = ?");
            $stmt->execute([$user_id]);
            $actions = $stmt->fetch(PDO::FETCH_ASSOC);
        }
        
        // [FIX] Explicitly cast boolean-like fields to integers for a consistent API response
        $actions['isCamRecording'] = (int)$actions['isCamRecording'];
        $actions['isVoiceRecording'] = (int)$actions['isVoiceRecording'];
        $actions['isLocationLive'] = (int)$actions['isLocationLive'];
        $actions['isOnline'] = (int)$actions['isOnline'];

        sendResponse(['success' => true, 'actions' => $actions]);
        
    } catch (Exception $e) {
        logError('Error getting user actions: ' . $e->getMessage());
        sendResponse(['error' => 'Failed to get user actions'], 500);
    }
}

function updateUserActions() {
    try {
        $data = json_decode(file_get_contents('php://input'), true);
        
        $headers = getallheaders();
         if ($headers === false) {
             $headers = [];
             foreach ($_SERVER as $name => $value) {
                if (substr($name, 0, 5) == 'HTTP_') {
                    $headerKey = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($name, 5)))));
                    $headers[$headerKey] = $value;
                }
            }
        }
        
        $user_id = getUserIdFromHeaders($headers);
        
        if (!$user_id) {
            sendResponse(['error' => 'User not authenticated'], 401);
            return;
        }
        
        if (!$data) {
            sendResponse(['error' => 'Invalid request data'], 400);
            return;
        }
        
        $database = new Database();
        $pdo = $database->getConnection();
        
        $updateFields = [];
        $values = [];
        
        if (isset($data['isCamRecording'])) {
            $updateFields[] = 'isCamRecording = ?';
            // [FIX] Ensure value is always 0 or 1
            $values[] = (int)(bool)$data['isCamRecording'];
        }
        
        if (isset($data['isVoiceRecording'])) {
            $updateFields[] = 'isVoiceRecording = ?';
            $values[] = (int)(bool)$data['isVoiceRecording'];
        }
        
        if (isset($data['isLocationLive'])) {
            $updateFields[] = 'isLocationLive = ?';
            $values[] = (int)(bool)$data['isLocationLive'];
        }
        
        if (isset($data['isOnline'])) {
            $updateFields[] = 'isOnline = ?';
            $values[] = (int)(bool)$data['isOnline'];
        }
        
        if (empty($updateFields)) {
            sendResponse(['error' => 'No valid fields to update'], 400);
            return;
        }
        
        $updateFields[] = 'updated_at = NOW()';
        
        $sql = "UPDATE actions SET " . implode(', ', $updateFields) . " WHERE user_id = ?";
        $values[] = $user_id;

        $stmt = $pdo->prepare($sql);
        $stmt->execute($values);
        
        sendResponse(['success' => true, 'message' => 'Actions updated successfully']);
        
    } catch (Exception $e) {
        logError('Error updating user actions: ' . $e->getMessage());
        sendResponse(['error' => 'Failed to update user actions'], 500);
    }
}

function updateLocationTracking() {
    try {
        $data = json_decode(file_get_contents('php://input'), true);
        
        $headers = getallheaders();
        if ($headers === false) {
             $headers = [];
             foreach ($_SERVER as $name => $value) {
                if (substr($name, 0, 5) == 'HTTP_') {
                    $headerKey = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($name, 5)))));
                    $headers[$headerKey] = $value;
                }
            }
        }

        $user_id = getUserIdFromHeaders($headers);
        
        if (!$user_id) {
            sendResponse(['error' => 'User not authenticated'], 401);
            return;
        }
        
        if (!$data || !isset($data['latitude']) || !isset($data['longitude'])) {
            sendResponse(['error' => 'Latitude and longitude are required'], 400);
            return;
        }
        
        $latitude = floatval($data['latitude']);
        $longitude = floatval($data['longitude']);
        
        if ($latitude < -90 || $latitude > 90 || $longitude < -180 || $longitude > 180) {
            sendResponse(['error' => 'Invalid coordinates'], 400);
            return;
        }
        
        $database = new Database();
        $pdo = $database->getConnection();
        
        $stmt = $pdo->prepare("UPDATE actions SET latitude = ?, longitude = ?, location_updated_at = NOW(), updated_at = NOW() WHERE user_id = ?");
        $stmt->execute([$latitude, $longitude, $user_id]);
        
        sendResponse(['success' => true, 'message' => 'Location updated successfully']);
        
    } catch (Exception $e) {
        logError('Error updating location: ' . $e->getMessage());
        sendResponse(['error' => 'Failed to update location'], 500);
    }
}

function getUserIdFromHeaders($headers) {
    $token = null;
    $authHeaderKey = array_filter(array_keys($headers), function($key) {
        return strtolower($key) === 'authorization' || strtolower($key) === 'x-auth-token';
    });

    if (!empty($authHeaderKey)) {
        $key = reset($authHeaderKey);
        $token = str_replace('Bearer ', '', $headers[$key]);
    }
    
    if ($token) {
        $parts = explode('_', $token);
        if (count($parts) >= 2) {
            return intval($parts[1]);
        }
    }
    return null;
}
?>
